<?php
declare(strict_types=1);

require_once __DIR__ . '/../common/bootstrap.php';
require_method('POST');

$input = get_json_input();
$tableNumber = clean_string($input['table_number'] ?? '', 20);
$items = $input['items'] ?? [];
$specialNotes = clean_string($input['special_notes'] ?? '', 255);

if ($tableNumber === '' || !is_array($items) || count($items) === 0) {
    json_error('Table number and at least one item are required', 422);
}

$tableStmt = $pdo->prepare('SELECT id, restaurant_id, status FROM tables WHERE table_number = :table_number LIMIT 1');
$tableStmt->execute([':table_number' => $tableNumber]);
$table = $tableStmt->fetch();
if (!$table) {
    json_error('Invalid table number', 404);
}

$tableId = (int) $table['id'];
$restaurantId = (int) $table['restaurant_id'];

$activeOrderStmt = $pdo->prepare("SELECT id FROM orders WHERE table_id = :table_id AND status IN ('received','preparing','ready','served') ORDER BY id DESC LIMIT 1");
$activeOrderStmt->execute([':table_id' => $tableId]);
$activeOrder = $activeOrderStmt->fetch();

$pdo->beginTransaction();
try {
    if ($activeOrder) {
        $orderId = (int) $activeOrder['id'];
    } else {
        $insertOrder = $pdo->prepare('INSERT INTO orders (restaurant_id, table_id, special_notes, status) VALUES (:restaurant_id, :table_id, :special_notes, :status)');
        $insertOrder->execute([
            ':restaurant_id' => $restaurantId,
            ':table_id' => $tableId,
            ':special_notes' => $specialNotes,
            ':status' => 'received'
        ]);
        $orderId = (int) $pdo->lastInsertId();
    }

    foreach ($items as $item) {
        $menuItemId = (int) ($item['menu_item_id'] ?? 0);
        $quantity = (int) ($item['quantity'] ?? 0);
        $itemNote = clean_string($item['special_notes'] ?? '', 255);

        if ($menuItemId <= 0 || $quantity <= 0) {
            throw new RuntimeException('Invalid item or quantity');
        }

        $menuStmt = $pdo->prepare('SELECT id, name, price, is_active, availability_state FROM menu_items WHERE id = :id AND restaurant_id = :restaurant_id LIMIT 1');
        $menuStmt->execute([':id' => $menuItemId, ':restaurant_id' => $restaurantId]);
        $menu = $menuStmt->fetch();
        if (!$menu || (int) $menu['is_active'] !== 1 || $menu['availability_state'] === 'out_of_stock') {
            throw new RuntimeException('Item unavailable: ' . $menuItemId);
        }

        $unitPrice = (float) $menu['price'];
        $lineTotal = round($unitPrice * $quantity, 2);

        $ins = $pdo->prepare('INSERT INTO order_items (order_id, menu_item_id, item_name_snapshot, unit_price_snapshot, quantity, line_total, special_notes, status)
                              VALUES (:order_id, :menu_item_id, :item_name, :unit_price, :quantity, :line_total, :special_notes, :status)');
        $ins->execute([
            ':order_id' => $orderId,
            ':menu_item_id' => $menuItemId,
            ':item_name' => $menu['name'],
            ':unit_price' => $unitPrice,
            ':quantity' => $quantity,
            ':line_total' => $lineTotal,
            ':special_notes' => $itemNote,
            ':status' => 'received'
        ]);
    }

    $updateTable = $pdo->prepare("UPDATE tables SET status = 'occupied' WHERE id = :table_id");
    $updateTable->execute([':table_id' => $tableId]);

    recalculate_bill($pdo, $orderId);

    $pdo->commit();
} catch (Throwable $e) {
    $pdo->rollBack();
    json_error($e->getMessage(), 422);
}

json_success([
    'order_id' => $orderId,
    'table_id' => $tableId,
    'message' => 'Order placed successfully'
]);